<?php
/**
 * Wrapper for Bluga WebThumb API, using Zend Framework.
 * Written by Cristi Nicolescu
 * http://mandagreen.com/code/webthumb/
 *
 *
 * Orignal wrapper for the WebThumb API by Joshua Eichorn
 * http://bluga.net/webthumb/
 *
 * Using this this wrapper requires Zend and Oxygen folders in your include path
 * You are free to rename the Oxygen namespace with your own namespace
 *
 * @author Cristi Nicolescu <cristi@mandagreen.com>
 * @author Cal Evans <cal@zend.com>
 * @author Joshua Eichorn <josh@bluga.net>
 * @copyright 2006 Cristi Nicolescu, Cal Evans, Joshua Eichorn
 * @license lgpl v2.1
 * @version 0.1
 */


/**
 * @see Oxygen_Service_Bluga_Webthumb_Exception
 */
require_once('Oxygen/Service/Bluga/Webthumb/Exception.php');


/**
 * @see Oxygen_Service_Bluga_Webthumb_Job
 */
require_once('Oxygen/Service/Bluga/Webthumb/Job.php');



class Oxygen_Service_Bluga_Webthumb {
	/**
	 * Jobs that are going to be submitted
	 */
	private $jobs = array();

	/**
	 * the webthumb api key.
	 *
	 * @var string 32 char md5 hash
	 */
	private $apiKey;

	/**
	 * Zend_Rest_Client request object
	 */
	private $restClient = null;

	/**
	 * Enable extra debug message
	 */
	private $debug = false;

	/**
	 * The url of the webthumb api base url
	 *
	 * @var string
	 */
	protected $webthumbApiBase     = 'http://webthumb.bluga.net';

	/**
	 * The url of the webthumb api endpoint
	 *
	 * @var string
	 */
	protected $webthumbApiEndPoint = '/api.php';

	/**
	 * the base URL to download the images from.
	 *
	 * @var string
	 */
	protected $imageUrl = 'http://webthumb.bluga.net/data/';

	/**
	 * The api version to send to the endpoint
	 */
	protected $apiVersion = 2;

	/**
	 * constructor
	 *
	 */
	function __construct($apikey = null)
	{
		if (!is_null($apikey)) {
			$this->apiKey = $apikey;
		}
	} // function __construct()


	/**
	 * set the webthumb api key.
	 *
	 * @param unknown_type $newKey
	 */
	public function setApiKey($newKey)
	{
		if (is_string($newKey)) {
			$this->apiKey = $newKey;
		} // if (is_string($newKey))
	} // public function setApiKey($newKey)


	/**
	 * add a url to the stack and set the properties for the request.
	 *
	 * @param String  $url
	 * @param string  $size
	 * @param Integer $width
	 * @param Integer $height
	 * @return bool
	 */
	public function addUrl($url, $size='small', $width=300, $height=300)
	{
	if (!$this->validateUrl($url)) {
		throw new Oxygen_Service_Bluga_Webthumb_Exception('Invalid Url: ' . $url);
	}

	$job = new Oxygen_Service_Bluga_Webthumb_Job();
	$job->options->width = $width;
	$job->options->height = $height;
	$job->options->url = $url;

	$job->size = $size;

	$this->jobs[$job->options->url] = $job;

		return $job;
	} // public function addUrl($url, $size='small',$width=300, $height=300)

	/**
	 * Add a job object to the request pool
	 */
	public function addJob(Oxygen_Service_Bluga_Webthumb_Job $job) {
		$this->jobs[$job->options->url] = $job;
	}


	/**
	 * delete a url from the stack
	 *
	 * @param String $url
	 * @return boolean
	 */
	public function deleteUrl($url)
	{
		$returnValue = false;
		if (array_key_exists($url, $this->jobs)) {
			unset($this->urlsjobs[$url]);
			$returnValue = true;
		} // if (array_key_exists($url, $this->urlsToImage))
		return $returnValue;
	} // public function deleteUrl($url)


	/**
	 * submit the queued URLS to webthumb.
	 * @throws Oxygen_Service_Bluga_Webthumb_Exception
	 *
	 */
	public function submitRequests()
	{
		if( count($this->jobs) < 1 ) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('No URLSs to image');
		}

		/**
		 * @see Oxygen_Service_Bluga_Webthumb_Request_Submit
		 */
		require_once('Oxygen/Service/Bluga/Webthumb/Request/Submit.php');

		$request = new Oxygen_Service_Bluga_Webthumb_Request_Submit();
		$request->apikey = $this->apiKey;
		$request->jobs = $this->jobs;
		$request->apiversion = $this->apiVersion;

		$payload  = $request->asXml();
		$response = $this->_transmitRequest($payload);

		$this->_parseSubmitResponse($response);
		return;
	} // public function submitRequests()


	/**
	 * prepares and sends the check status payload for all submitted jobs.
	 *
	 */
	public function checkJobStatus()
	{
		if (count($this->jobs)<1) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('No jobs to check');
		} // if (count($this->urlsToImage)<1)

		/**
		 * @see Oxygen_Service_Bluga_Webthumb_Request_Status
		 */
		require_once('Oxygen/Service/Bluga/Webthumb/Request/Status.php');

		$status = new Oxygen_Service_Bluga_Webthumb_Request_Status();
		$status->apikey = $this->apiKey;
		foreach($this->jobs as $job) {
			$status->jobs[] = $job->status->id;
		}
		$xml = $status->asXML();

		if (strlen($xml)>0) {
			$response = $this->_transmitRequest($xml);
			$this->_parseStatusResponse($response);
		} // if (strlen($xml)<1)
		return;
	} // public function checkJobStatus()

	/**
	 * handles the actual transmission of the payload.
	 *
	 * @param unknown_type $request
	 * @return Zend_Http_Response the response from the server.
	 */
	protected function _transmitRequest($request)
	{
		/**
		 * @var Zend_Rest_Client $client
		 */
		$client = $this->getRestClient()->resetParameters();

		/**
		 * @var Zend_Http_Response $response
		 */
		$response = $client->restPost($this->webthumbApiEndPoint, $request);
		if ($response->isError()) {
			if ($this->debug)
			{
				echo "We had an error the response object is\n";
				echo "#################\n";
				echo var_dump($response);
				echo "#################\n";
			}
			throw new Oxygen_Service_Bluga_Webthumb_Exception('API didn\'t respond with a 200 status code (status code: '.$response->getStatus().')');
		}

		if ($this->debug) {
			echo "Request Body is\n";
			echo "#################\n";
			echo $response->getBody();
			echo "#################\n\n";
		}

		return $response;
	} // protected function _transmitRequest($request)


	/**
	 * Takes the response from submitting urls for imaging and parses it out.
	 * Places the responses in the appropriate places in the urlsToImage
	 * array.
	 *
	 * @param Zend_Http_Response $response the response to process
	 *
	 */
	protected function _parseSubmitResponse(Zend_Http_Response $response)
	{
		$this->_checkContentType($response);

		if ($this->debug) {
			echo "Submit Response Body is\n";
			echo "#################\n";
			echo $response->getBody();
			echo "#################\n";
		}

		libxml_use_internal_errors(true);
		$xml = new SimpleXMLElement($response->getBody());

		$err = libxml_get_last_error();
		if ($err) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('Invalid XML Payload Returned: '.$err->getMessage());
		}

		// version 2 api returns errors as xml check for an xml error message
		if (isset($xml->error))
		{
			throw new Oxygen_Service_Bluga_Webthumb_Exception('Error message returned from API endpoint: '.$xml->error.'('.$xml->error['type'].')');
		}

		if (!isset($xml->jobs))
		{
			var_dump($xml);
			throw new Oxygen_Service_Bluga_Webthumb_Exception('Unknown response');
		}

		foreach ($xml->jobs->job as $thisJob) {
			$thisUrl = (string)$thisJob['url'];
			$this->jobs[$thisUrl]->status->id = (string)$thisJob;
			$this->jobs[$thisUrl]->status->start_time = (string)$thisJob['time'];
			$this->jobs[$thisUrl]->status->est_time = (string)$thisJob['estimate'];
			$this->jobs[$thisUrl]->status->cost = (string)$thisJob['cost'];
			$this->jobs[$thisUrl]->status->id = (string)$thisJob;
			$this->jobs[$thisUrl]->status->status = 'Transmited';
		} // foreach ($xml->jobs as $thisJob)

		return;
	} // protected function _parseSubmitResponse($response)


	/**
	 * Takes the response from a status check request and parses it out. Places
	 * the relevant information in the urlsToImage array.
	 *
	 * @param Zend_Http_Response $response the response to process.
	 *
	 */
	protected function _parseStatusResponse(Zend_Http_Response $response)
	{
		// this throws an Oxygen_Service_Bluga_Webthumb_Exception
		$this->_checkContentType($response);

		if ($this->debug) {
			echo "Status Response Body is\n";
			echo "#################\n";
			echo $response->getBody();
			echo "#################\n";
		}

		libxml_use_internal_errors(true);
		$xml = new SimpleXMLElement($response->getBody());
		$err = libxml_get_last_error();
		if ($err) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('Invalid XML Payload Returned: '.$err->getMessage());
		}

		// version 2 api returns errors as xml check for an xml error message
		if (isset($xml->error))
		{
			$error = $xml->error;
			if (count($error) > 1)
			{
				$error = $xml->error[1];
			}
			throw new Oxygen_Service_Bluga_Webthumb_Exception('Error message returned from API endpoint: '.$error.'('.$error['type'].')');
		}

		if (!isset($xml->jobStatus))
		{
			throw new Oxygen_Service_Bluga_Webthumb_Exception('Unknown response to a job status request');
		}

		foreach ($xml->jobStatus->status as $thisJob) {
			$thisId	  = (string)$thisJob['id'];
			$thisUrl	 = $this->_findUrlByJobId($thisId);

			$this->jobs[$thisUrl]->status->status = (string)$thisJob;
			$this->jobs[$thisUrl]->status->pickup = (string)$thisJob->status['pickup'];
		} // foreach ($xml->jobs as $thisJob)

		return;
	} // protected function _parseStatusResponse()


	/**
	 * finds the Content-Type in the response.
	 *   XML  == Good
	 *   HTML == Error.
	 *
	 * @param Zend_Http_Response $response
	 * @return boolean
	 * @throws Oxygen_Service_Bluga_Webthumb_Exception
	 */
	protected function _checkContentType(Zend_Http_Response $response)
	{
		$contentType = strtolower($response->getHeader('Content-Type'));
		if ( !$contentType ) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('No Content-Type in response.');
		}
		if ($contentType != 'text/xml') {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('There was an error. Content-Type returned was '.$contentType."\n");
		}

		return true;
	} // protected function _checkContentType($response)


	/**
	 * Given a jobID, this finds the URL for it. Used in the status parser to
	 * match up the information to the proper array.
	 *
	 * @param string $jobId
	 * @return string The URL found.
	 */
	protected function _findUrlByJobId($jobId)
	{
		$thisUrl='';
		foreach($this->jobs as $url => $job) {
			if ($job->status->id === $jobId) {
				$thisUrl = $url;
				break;
			} // if ($params['job']===$job_id)
		} // foreach($this->urlsToImage as $url=>$params)
		return $thisUrl;
	}


	/**
	 * check to see if all images are ready for download.
	 *
	 * @return boolean
	 */
	public function readyToDownload()
	{
		$returnValue = true;
		foreach($this->jobs as $job) {
			$returnValue = ($returnValue AND $job->status->status==='Complete');
		} // foreach($this->urlsToImage as $url=>$params)
		return $returnValue;
	} // public function readyToDownload()


	/**
	 * If all images are complete then it will send a request for each image.
	 *
	 */
	public function fetchAll($dir = null)
	{
		if (count($this->jobs)<1) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('No URLSs to image');
		} // if (count($this->urlsToImage)<1)

		if (!$this->readyToDownload()) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('No images ready to download.');
		} // if (!$this->readyToDownload())

		foreach ($this->jobs as $job) {
			if($job->status->status == 'Complete') {
				$this->fetchToFile($job, null, null, $dir);
			} // if($this->urlsToImage['status'==='Complete'])
		} // foreach ($this->urlsToImage as $params)

		return;
	} // public function fetchAll()


	/**
	 * fetches the given jobid and stores it on the filesystem in the filename
	 * specified.
	 *
	 * @param string $job
	 * @param string $filename
	 * @param string $size
	 *
	 */
	public function fetchToFile(Oxygen_Service_Bluga_Webthumb_Job $job, $filename=null, $size=null, $outDir = null )
	{
		if (is_null($filename)) {
			if (isset($job->file)) {
				$filename = $job->file;
			}
			else {
				$filename = $job->status->id;
			}
			$ext = 'jpg';
			if ($job->options->outputType == 'png') {
				$ext = 'png';
			}
			$filename .= '.'.$ext;
		}

		if (!is_null($outDir)) {
			$filename = rtrim($outDir, '/') . '/' . $filename;
		}

		if (is_null($size)) {
			$size = 'small';
			if (isset($job->size)) {
				$size = $job->size;
			}
		}

		/**
		 * @see Oxygen_Service_Bluga_Webthumb_Request_Fetch
		 */
		require_once('Oxygen/Service/Bluga/Webthumb/Request/Fetch.php');

		$request = new Oxygen_Service_Bluga_Webthumb_Request_Fetch($job, $size);
		$request->apikey = $this->apiKey;
		$payload = $request->asXML();

		$image   = $this->_transmitRequest($payload);
		file_put_contents($filename, $image->getBody());
		return;
	} // public function fetchToFile($job='', $filename='', $size='small' )

	public function validateUrl( $url )
	{
		require_once('Zend/Uri.php');
		try {
			$uri = Zend_Uri::factory($url);
			return true;
		}
		catch( Exception $e1 ) {
			try {
				$uri = Zend_Uri::factory( 'http://' . $url );
				return true;
			}
			catch( Exception $e2 ) {
				return false;
			}
		}

		return false;
	}


	/**
	 * sets the baseDir
	 *
	 * @param string $newValue
	 * @return boolean
	 */
	public function setBaseDir($newValue = '')
	{
		$returnValue = false;

		if (!empty($newValue)) {
			$newValue = trim($newValue);
			$newValue .= substr($newValue, -1) != '/' ? '/' : '';
			$this->baseDir = $newValue;
		} // if (!empty($newValue))

		return $returnValue;
	} // public function setBaseDir($newValue='')


	/**
	 * retrieves the baseDir
	 *
	 * @return string
	 */
	public function getBaseDir()
	{
		return $this->baseDir;
	} // public function getBaseDir()


	/**
	 * create and return the REST client
	 *
	 * @return Zend_Rest_Client the REST client to use
	 */
	public function getRestClient() {
		if( is_null($this->restClient) ) {
			require_once('Zend/Rest/Client.php');
			$this->restClient = new Zend_Rest_Client($this->webthumbApiBase);
		}

		return $this->restClient;
	}


	/**
	 * return current credit status
	 *
	 * @return array details of current credit status
	 */
	public function checkCreditStatus() {
		/**
		 * @see Oxygen_Service_Bluga_Webthumb_Request_Credits
		 */
		require_once('Oxygen/Service/Bluga/Webthumb/Request/Credits.php');

		$request = new Oxygen_Service_Bluga_Webthumb_Request_Credits();
		$request->apikey = $this->apiKey;
		$payload = $request->asXML();

		$client = $this->getRestClient()->resetParameters();
		$response = $client->restPost($this->webthumbApiEndPoint, $payload);

		libxml_use_internal_errors(true);
		$xml = new SimpleXMLElement($response->getBody());

		$err = libxml_get_last_error();
		if( $err ) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('Invalid XML Payload Returned: ' . $err->getMessage());
		}

		if( !isset($xml->credits) ) {
			throw new Oxygen_Service_Bluga_Webthumb_Exception('Invalid XML Payload Returned - CREDITS node not found');
		}

		$ret = array();
		foreach( $xml->credits->children() as $node => $value ) {
			$ret[ (string)$node ] = (string)$value;
		}

		return $ret;
	}

}// class Oxygen_Service_Bluga_Webthumb
/* vim: set expandtab tabstop=2 shiftwidth=4 softtabstop=2: */
